/*
Copyright 2008-2011 Broadcom Corporation

This program isthe proprietary software of Broadcom Corporation and/or its
licensors, and may only be used, duplicated, modifiedor distributed pursuant to
the terms and conditions of a separate, written license agreement executed
between you and Broadcom (an "Authorized License"). Except as set forth in an
Authorized License, Broadcom grants no license (express or implied), right to
use, or waiver of any kind with respect to the Software, and Broadcom expressly
reserves all rights in and to the Software and all intellectual property rights
therein. IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU HAVE NO RIGHT TO USE THIS
SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY NOTIFY BROADCOM AND DISCONTINUE ALL
USE OF THE SOFTWARE.

Except as expressly set forth in the Authorized License,

1. This program, including its structure, sequence and organization, constitutes
   the valuable trade secrets of Broadcom, and you shall use all reasonable
   efforts to protect the confidentiality thereof, and to use this information
   only in connection with your use of Broadcom integrated circuit products.

2.TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWAREIS PROVIDED "AS IS" AND
   WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
   WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
   THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMSANY AND ALL IMPLIED WARRANTIES
   OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
   LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
   OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
   USE OR PERFORMANCE OF THE SOFTWARE.

3.TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
   LICENSORS BE LIABLE FOR(i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
   EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
   USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
   THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
   ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
   LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
   ANY LIMITED REMEDY.
*/
/** \file framework.h
 *
 * \brief This are the framework definitions
 *
 **************************************************/

#ifndef FRAMEWORK_H_
#define FRAMEWORK_H_

/***************************************************
*                 Include section
***************************************************/
#include "l2_driver.h"

/***************************************************
 *                 Public Defines Section
 ***************************************************/

/***************************************************
 *                 Public Constants Section
 ***************************************************/

/** \brief Byte size of a protocol packet */
#define MAX_LEN   1520

/***************************************************
 *                 Public Typedefs Section
 ***************************************************/

/** \brief     Simplify buffer management in protocol stacks by supporting a
 *             buffer that allows addition and extraction of different protocol
 *             layers.
 */
typedef struct st_pkt
{
  TU8 srcMAC[6];       //!< packet source mac address
  TU8 data[MAX_LEN];   //!< packet data
  TU16 datalen;        //!< packet data length
  TU16 body_offset;    //!< body start point
	TU16 mmtype;         //!< MMTYPE for this protocol pkt
	TU16 timeout;        //!< timeout that the L2 layer applies to this pkt
  TU8 retries;         //!< number of retries that the L2 layer applies to this pkt
  TU8 msg_id;          //!< message id
  TU8 mmv;             //!< MMV supported
  TU8 FMI;             //!< Fragmentation Management Information
  TU8 fmsn;            //!< Fragmentation Message Sequence Number
  TU8* fchunk;         //!< Chunk of bytes containing fragmented pkt data
  TU16 fchunk_len;     //!< Chunk of bytes containing fragmented pkt data length
}
tS_pkt;

/** \brief     Defines the individual layer interface
 */
typedef struct st_layer
{
   void (*Transmit)(struct st_layer *, tS_pkt *); /**< Process and transmit the
                                                packet passed by higher layer.*/
   void (*Handle_Receive)(struct st_layer *, tS_pkt *); /**< Receive and process
                                          the packets passed by a lower layer.*/
   struct st_layer *p_upperlayer; //!< point to the upper layer
   struct st_layer *p_lowerlayer; //!< point to the lower layer
}
tS_layer;

/** \brief     Defines a protocol stack */
typedef struct st_stack
{
   tS_layer *p_highest; //!< point to the highest layer.
   tS_layer *p_lowest;  //!< point to the lowest layer.
}
tS_stack;

/***************************************************
 *         Public Function Prototypes Section
 ***************************************************/

/**
 * \brief            Destroy a protocol stack
 *
 * \param p_stack    (in) protocol stack to destroy
 */
void _stack_destroy(tS_stack* p_stack);

/**
 * \brief            Send a protocol packet using a protocol stack
 *
 * \param p_stack    (in) protocol stack
 * \param p_pkt      (in) protocol packet to send
 */
void _stack_sendpkt(tS_stack* p_stack, tS_pkt* p_pkt);

/**
 * \brief            Protocol packet received from a network device
 *
 * \param p_stack    (in) protocol stack
 * \param p_pkt      (in) protocol packet to send
 */
void _stack_pktrecv(tS_stack* p_stack, tS_pkt* p_pkt);

/**
 * \brief               Create a protocol packet from a stream of bytes.
 *                      This function is used when data is received from the
 *                      network and is injected into the protocol stack.
 *
 * \param p_data        (in) pointer to a stream of bytes
 * \param datalen       (in) number of bytes in the stream passed
 * \param srcMAC        (in) pkt source MAC address
 *
 * \return tS_pkt*      Created packet
 */
tS_pkt* _pkt_create(TU8* p_data, int datalen, TU8* srcMAC);

/**
 * \brief               Create a protocol packet. This packet will hold a chunk
 *                      of bytes from fragmented packets.
 *
 * \param p_data        (in) pointer to chunk of bytes
 * \param datalen       (in) number of bytes in the chunk
 * \param p_frag_pkt    (in) fragment packet
 *
 * \return tS_pkt*      Created packet
 */
tS_pkt* _pkt_create_fragpkt(TU8* p_data, int datalen, tS_pkt* p_frag_pkt);

/**
 * \brief            Create a protocol packet from a mmtype
 *                   This function is used when an app needs to send a L2
 *                   primitive using the procol stack.
 *
 * \param mmtype     (in) mmtype for the packet
 *
 * \return tS_pkt*   Created packet
 */
tS_pkt* _pkt_primitive(TU16 mmtype);

/**
 * \brief            Add a stream of bytes to a protocol packet's header
 *
 * \param p_pkt      (in) protocol packet
 * \param p_data     (in) pointer to a stream of bytes
 * \param datalen    (in) number of bytes in the stream passed
 */
void _pkt_addheader(tS_pkt* p_pkt, TU8* p_data, int datalen);

/**
 * \brief            Add a byte to a protocol packet's header
 *
 * \param p_pkt      (in) protocol packet
 * \param u8         (in) byte
 */
void _pkt_addheaderU8(tS_pkt* p_pkt, TU8 u8);

/**
 * \brief            Add two bytes to a protocol packet's header
 *
 * \param p_pkt      (in) protocol packet
 * \param u16        (in) unsigned short
 */
void _pkt_addheaderU16(tS_pkt* p_pkt, TU16 u16);

/**
 * \brief            Add a stream of bytes to a protocol packet's payload
 *
 * \param p_pkt      (in) protocol packet
 * \param p_data     (in) pointer to a stream of bytes
 * \param datalen    (in) number of bytes in the stream passed
 */
void _pkt_addbody(tS_pkt* p_pkt, TU8* p_data, int datalen);


/**
 * \brief            Add a byte to a protocol packet's payload
 *
 * \param p_pkt      (in) protocol packet
 * \param u8         (in) byte
 */
void _pkt_addbodyU8(tS_pkt* p_pkt, TU8 u8);

/**
 * \brief            Add two bytes to a protocol packet's payload
 *
 * \param p_pkt      (in) protocol packet
 * \param u16        (in) unsigned short
 */
void _pkt_addbodyU16(tS_pkt* p_pkt, TU16 u16);

/**
 * \brief            Add four bytes to a protocol packet's payload
 *
 * \param p_pkt      (in) protocol packet
 * \param u32        (in) unsigned int
 */
void _pkt_addbodyU32(tS_pkt* p_pkt, TU32 u32);

/**
 * \brief            Add four bytes to a protocol packet's trailer
 *
 * \param p_pkt      (in) protocol packet
 * \param u32        (in) unsigned int
 */
void _pkt_addtraileru32(tS_pkt* p_pkt, TU32 u32);

/**
 * \brief            Add a byte to a protocol packet's trailer
 *
 * \param p_pkt      (in) protocol packet
 * \param u8         (in) byte
 */
void _pkt_addtraileru8(tS_pkt* p_pkt, TU8 u8);

/**
 * \brief            Extract to a stream of bytes a protocol packet's header
 *
 * \param p_pkt      (in)  protocol packet
 * \param p_data     (out) pointer to a stream of bytes
 * \param datalen    (in)  stream of bytes size
 */
void _pkt_extractheader(tS_pkt* p_pkt, TU8* p_data, int datalen);

/**
 * \brief            Extract a byte to a protocol packet's header
 *
 * \param p_pkt      (in) protocol packet
 * \param p_u8       (out) point to a byte which will hold the data extracted
 */
void _pkt_extractheaderU8(tS_pkt* p_pkt, TU8* p_u8);

/**
 * \brief            Extract two bytes to a protocol packet's header
 *
 * \param p_pkt      (in)  protocol packet
 * \param p_u16      (out) point to a u_short which will hold the data extracted
 */
void _pkt_extractheaderU16(tS_pkt* p_pkt, TU16* p_u16);

/**
 * \brief            Extract to a stream of bytes a protocol packet's trailer
 *
 * \param p_pkt      (in)  protocol packet
 * \param p_data     (out) pointer to a stream of bytes
 * \param datalen    (in)  stream of bytes size
 */
void _pkt_extracttrailer(tS_pkt* p_pkt, TU8* p_data, int datalen);

/**
 * \brief            Extract to a stream of bytes a protocol packet's payload
 *
 * \param p_pkt      (in)  protocol packet
 * \param p_data     (out) pointer to a stream of bytes
 * \param p_size     (out) stream of bytes size
 */
void _pkt_getbody(tS_pkt* p_pkt, TU8* p_data, int* p_size);

/**
 * \brief            Discard a protocol packet (deallocate memo resources)
 *
 * \param p_pkt      (in) protocol packet
 */
void _pkt_discard(tS_pkt* p_pkt);

/**
 * \brief            Duplicate a protocol packet
 *
 * \param p_pkt      (in) protocol packet
 *
 * \return tS_pkt*   Created packet
 */
tS_pkt* _pkt_dup(tS_pkt* p_pkt);

/**
 * \brief           Print packet content
 */
void _pkt_print(tS_pkt* p_pkt);

/**
 * \brief            Extract the two first bytes of a stream and packed into
 *                   an u_short
 *
 * \param p_data     (in) Stream of bytes
 */
TU16 _packU16(TU8* p_data);

/**
 * \brief            Extract the four first bytes of a stream and packed into
 *                   an TU32
 *
 * \param p_data     (in) Stream of bytes
 */
TU32 _packU32(TU8* p_data);

/**
 * \brief            Extract the four first bytes of a stream and packed into
 *                   an TU24
 *
 * \param p_data     (in) Stream of bytes
 */
TU32 _packU24(TU8* p_data);

/**
 * \brief            Extract the eight first bytes of a stream and packed into
 *                   an T64
 *
 * \param p_data     (in) Stream of bytes
 */
T64 _packU64(TU8* p_data);

#endif
